%	 Copyright (C) 2008  Frhwirth-Schnatter
%	 Copyright (C) 2011  Bluder, Plankensteiner
%
%    This program is free software: you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation, either version 3 of the License, or
%    (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program.  If not, see <http://www.gnu.org/licenses/>.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DEMONSTRATION OF LINEAR REGRESSION MODELLING BASED MIXTURES-OF-EXPERTS %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%% NORMAL DISTRIBUTIONS %%%%%%%%%%%%%%%%%%%%%%%%%% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Obligatory fields of data 
%%% .y: 1 x N data vector 
%%% .N: number of data
%%% .X: (d+1) x N matrix with covariate values and leadind 1
%%% .censor: 1 x N vector with 0 (datapoint uncensored) or 1 (datapoint censored)
%%% .Xweights: K x N matrix of cofactors for mixture weights model 
%%% .weightfun: name of function for mixture weights model

%%  EmpEst_weights & EmpEst_weightfun vs Cofactor_weights & Cofactor_weightfun

%%% .EmpEst weights are the empirical estimated weights for the data (use in combination with function EmpEst_weightfun)
%%% .Cofactor_weights is the matrix of cofactors for mixture weights
%%% regression (use in combination with function Cofactor_weightfun)
 

%% Load data:
clear all;
load('datareg_MoE.mat');

data=datareg;

%% SET PARAMETERS - these parameters have to be defined by the user
K=2; % data is a mixture of K=2 distributions
M= 5000; % number of MCMC draws after burnin
burnin=1000; % burnin

name = 'mcmcout'; %define a name for the model 
comment = 'comment?'; % add a comment

%% DEFINE MODEL:
model.K=K;
model.dist='Normal';
model.d=size(data.X,1);

%% USE either empirical estimated weights OR the modeled weights for MOE model

%%% empirical estimated weights (more popular: for model development use empirical estimated weights)
data.weightfun = data.EmpEst_weightfun;
data.Xweights = data.EmpEst_weights;

%%% OR modeled weights (it was shown that mixture weights can be modeled with a
%%% beta distribution function depending on two cofactors)
data.weightfun = data.Cofactor_weightfun; % regression model for mixture weights
data.Xweights = data.Cofactor_weights; % cofactors for model

%% delete unused variables
data = rmfield(data, 'EmpEst_weights');
data = rmfield(data, 'Cofactor_weights');
data = rmfield(data, 'EmpEst_weightfun');
data = rmfield(data, 'Cofactor_weightfun');

%% Definitions for MOE Model:

model.MOE=true; % MOE model will be used
model.weightfun=data.weightfun; % the name of the weight function (is stored in data.weightfun)
model.indicmod.dist = 'FixedWeights'; 

%% DEFINE PRIOR

%% Either use a default prior => a hierarchical normal prior: 
prior=priordefine_MOE(data, model);

%% OR
%%%% use a non hierarchical normal prior:
prior.hier = false;
prior=priordefine_MOE(data, model, prior);
 
%% INCLUDE PRIOR INFORMATION:
%%% AT THIS POINT A NORMAL PRIOR IS USED, because our prior information implies
%%% normal distribution N(mu, sigma^2):

%%%  we have no prior information for the intercept: b_{0,k}
%%%  we have prior information for b_{1,k}, b_{2,k} and b_{3,k}
%%%  
%%%  b_{1,k} ~ N(-1/3, 1);
%%%  b_{2,k} ~ N(-4.25, 0.44);
%%%  b_{3,k} ~ N(0.93, 0.25);
%%%   include the information in the defined prior:

prior.par.beta.b(2,:)=-1/3; 
prior.par.beta.b(3,:)=-4.25;
prior.par.beta.b(4,:)=0.93;
prior.par.beta.Binv(2,2,:)=1/1;
prior.par.beta.Binv(3,3,:)=1/0.44;
prior.par.beta.Binv(4,4,:)=1/0.25;  


%% DEFINE MCMC PROPERTIES:
[data, model, mcmc]= mcmcstart_MOE(data, model);
mcmc.M= M;
mcmc.burnin=burnin;
mcmc.storeS=M; % to check allocations afterward
     

%% RUN MCMC: - this can take some time (depending on sample size, size of data (censored/uncensored), and MCMC definitions)
mcmcout= mcmc_MOE(data, model, prior, mcmc); 


%% PLOT MCMC RESULTS: plots the posterior draw-based values e.g. mixture likelihood values, entropy etc. and the posterior draws of the parameters
mcmcplot_MOE(mcmcout)


%% CLASSIFY THE DATA BASED ON MCMC DRAWS & PLOT THE CLASSIFICATION PROBABILITIES & CLUSTERING RESULTS
clust= mcmcclust_MoE(data, mcmcout); % based on the posterior draws, the datapoints are clustered 
mcmcclustplot_MoE(data,clust); % now you can see the posterior allocation probabilities 


%% VALUE OF THE MARGINAL LIKELIHOOD:
[marlik,mcmcout]=mcmcbf_MOE(data,mcmcout); 

 
%% POSTERIOR ESTIMATES 
[est,mcmcout]=mcmcestimate_MOE(mcmcout);  

%% MODEL EVALUATION 

% significance level for KS-test
alpha = 0.05;
PIT=PIT_mixtures(data, mcmcout,alpha);  % if model assumption is accurate, PIT values are uniformly distributed
% the output gives you a summary about the PIT values statistic
% this is a real dataset...for our data the PIT values does not show that
% the model fits data well


%% GOODNESS OF FIT CRITERIA:
[ic,mcmcout]=mcmcic_MOE(data,mcmcout); % goodness of fit criteria are determined. Criteria are determined with the posterior draw, that gives the largest likelihood
    

%% STORE RESULTS:
mcmcout.data=data;
mcmcout.name= name;
mcmcout.comment = comment;
mcmcstore(mcmcout); 


%% PREDICTION:
%%% predict new data: 'data_future' and compare with real observations:
%%% obligatory fields: 
%%%        - X: cofactors for regression model
%%%        - Xweights: cofactors for mixture weight regression model

%%% for comparison: the real data (.y, .censor, N) are shown and compared with prediction

%%% FOR MOE: Define .Xweights and .weightfun 

data_future.weightfun= 'Beta_weight_fun';
data_future.Xweights = data_future.MOE_Xweights;
[fig,pred]=mcmcpreddens_MOE(data_future, mcmcout); % based on the posterior draws, the new dataset is predicted.
% in the graphic you can see the pdf and the cdf of empirical and predicted dataset 


%% PERFORM CROSSVALIDATION
%%% For model development: data.Xweights and data.weightfun are used
%%% if different Xweights and weightfuns for model development and prediction are used 
%%% (e.g.: Model development with empirical estimated weights and prediction with a MOE function)
%%% define model.MOE_pred = true, 
%%%    => model.MOE_Xweights and model.MOE_weightfun (for pred.) have to be defined additionally.

%%% Load & prepare data:
clear all;
load('datareg_MoE.mat');

data=datareg;
%%% Define model:
model.K=2;
model.dist='Normal'; 
model.d=size(data.X,1);
model.MOE=true; % MOE model will be used
model.indicmod.dist = 'FixedWeights'; 

% use emp.est.weights for model development
model.weightfun = data.EmpEst_weightfun;
data.Xweights = data.EmpEst_weights;
 
% use a regression model for the mixture weights (for data prediction)
model.MOE_pred=true;
model.MOE_Xweights= data.Cofactor_weights; % cofactors for model
model.MOE_weightfun=data.Cofactor_weightfun; % regression model for mixture weights

%%%% Define prior: 

%%% Either use a default prior => a hierarchical normal prior: 
prior=priordefine_MOE(data, model);

%%% OR
%%%% use a non hierarchical normal prior:
prior.hier = false;
prior=priordefine_MOE(data, model, prior);
 
%%% INCLUDE PRIOR INFORMATION:
%%% AT THIS POINT A NORMAL PRIOR IS USED, because our prior information implies
%%% normal distribution N(mu, sigma^2):

%%%  we have no prior information for the intercept: b_{0,k}
%%%  we have prior information for b_{1,k}, b_{2,k} and b_{3,k}
%%%  
%%%  b_{1,k} ~ N(-1/3, 1);
%%%  b_{2,k} ~ N(-4.25, 0.44);
%%%  b_{3,k} ~ N(0.93, 0.25);
%%%   include the information in the defined prior:

prior.par.beta.b(2,:)=-1/3; 
prior.par.beta.b(3,:)=-4.25;
prior.par.beta.b(4,:)=0.93;
prior.par.beta.Binv(2,2,:)=1/1;
prior.par.beta.Binv(3,3,:)=1/0.44;
prior.par.beta.Binv(4,4,:)=1/0.25;  


%%%% delete unused variables
clear data_future;
data = rmfield(data, 'EmpEst_weights');
data = rmfield(data, 'Cofactor_weights');
data = rmfield(data, 'EmpEst_weightfun');
data = rmfield(data, 'Cofactor_weightfun');

% perform cross validation
CrossVal=CrossValidation_MOE(data, 'mcmc_MOE', model, prior);