%	 Copyright (C) 2011  Bluder, Plankensteiner
%
%    This program is free software: you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation, either version 3 of the License, or
%    (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program.  If not, see <http://www.gnu.org/licenses/>.

function [res, Sperm, prior, post] = posteriorSS(data, model, prior, post, Sperm)
% computes posterior draws for the given data, model and prior

% method can only handle regression
% models or MoE models based on univariate Gaussian data, Dec.2011
  
%%%%%%%%%%% Input %%%%%%%%%%% 
%   data, model,prior, Sperm have the same structure as for the funtion 'mcmc_MOE'
%   data = structure array contaiing info about the data to be modeled
%   model = structure array defining the mixture model
%   prior = all prior information for the model
%   Sperm = additional info for slice sampler (e.g. to speed up)

%%%%%%%%%%% Output %%%%%%%%%%%
%   res = structure array with posterior draws for all parameters
%   Sperm = updated version of the input
%   prior = prior of input


if ~isfield(model,'K')
    K=1; model.K=1;
else
    K= model.K;
end

%      Normal or student mixture;
if ~isfield(model,'error')  % Normal or student mixture; default: switching variane
    model.error='switch';
end

if nargin==3
    % check the dimension of the data
    % data are handled as data stored by row
    if isfield(data,'bycolumn') ibycolumn=data.bycolumn; else ibycolumn=false; end  % ibycolumn true: data stored by column
    if ibycolumn
        data.y=data.y';
        if isfield(data,'X') data.X=data.X'; end
        data.bycolumn =false;
    end
    if ~isfield(data,'empty') data.empty=false; end
    if ~isfield(data,'N') data.N=size(data.y,2); end % optimized
    
    if and(model.K>1,~isfield(data,'S'))
        warn('The data have to be completely specified, including the allocations in field S');
        res=[];return;
    end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% calculations for normal disrtibutions

mufix=false;sigmafix=false;betafix=false;
if isfield(model,'parfix')
    if  isfield(model.parfix,'mu')    mufix=model.parfix.mu;     end
    if  isfield(model.parfix,'beta')  betafix=model.parfix.beta;     end
    if  isfield(model.parfix,'sigma') sigmafix=model.parfix.sigma;     end
end

if model.K==1   % single member from the distribution family
    S=ones(1,data.N);
else    % mixture distribution
    S=data.S;
end

if all([betafix sigmafix isfield(model,'d')])
    warn('It is not possible to fix both the regression parameters and the variance in function posterior');post=[];return;
    
elseif all([~betafix ~sigmafix isfield(model,'d')])
    % finite mixture of multiple regression models
    res = struct('beta',zeros(size(prior.par.beta.b)),'sigma',zeros(size(prior.par.sigma.c)));
    
    % posterior sampling for independence prior
    post.par.sigma = struct('c',nan(size(prior.par.sigma.c)),'C',nan(size(prior.par.sigma.C)));
    post.par.beta = struct('b',nan(size(prior.par.beta.b)),'B',nan(size(prior.par.beta.Binv)),...
        'Binv',nan(size(prior.par.beta.Binv)));
    
    for k=1:model.K
        Xk = data.X(:,S==k)';
        yk = data.y(1,S==k)';
        
        censor = data.censor(:,S==k)';
        nocensor = all(~censor);
        
        b0 = prior.par.beta.b(:,k)';
        B0 = sqrt(inv(prior.par.beta.Binv(:,:,k)));
        priorBeta = @(beta) log(mvnpdf(beta,b0,B0));
        c0 = prior.par.sigma.c(k);
        C0 = prior.par.sigma.C(k);
        if prior.hier
            res.C0(k)=1/C0;
        end
        priorSigma= @(sigma) log(C0^c0/gamma(c0)*(1/sigma)^(c0+1)*exp(-C0/sigma));
        d = length(b0);
        
        if nocensor
            if model.df==0
                % simulate regression coefficients for a mixture of regression  models
                modreg=model; modreg.parfix.sigma=true;
                modreg.K=1; modreg.par.sigma = model.par.sigma(k);
                
                datareg = data; datareg.y = yk'; datareg.X = Xk';
                datareg.N = length(yk); datareg = rmfield(datareg,'S');
                
                prior1=prior.par.beta;
                prior1.b=b0';
                prior1.Binv=prior.par.beta.Binv(:,:,k);
                
                postbeta=posterior_MOE(datareg,modreg,prior1);
                post.par.beta.b(:,k) = postbeta.b;
                post.par.beta.B(:,:,k) = postbeta.B;
                post.par.beta.Binv(:,:,k) = postbeta.Binv;
                modreg.par.beta=prodnormultsim(struct('mu',postbeta.b,'sigma',postbeta.B));
                res.beta(:,k) = modreg.par.beta;
                
            else   %  mixed-effects mixture of regression  model
                res.error=true; return;
            end
            % simulate error variances
            if ~model.parfix.sigma
                modreg=model; modreg.parfix.beta=true;
                modreg.K=1; modreg.par.beta = res.beta(:,k);
                
                datareg = data; datareg.y = yk'; datareg.X = Xk';
                datareg.N = length(yk); datareg = rmfield(datareg,'S');
                
                prior1 = prior.par.sigma;
                prior1.c=c0;
                prior1.C=C0;
                if model.df>0 modreg.parfix.alpha=true; end
                postsigma =posterior_MOE(datareg,modreg,prior1);
                post.par.sigma.c(k) = postsigma.c;
                post.par.sigma.C(k) = postsigma.C;
                modreg.par.sigma=1./prodgamsim(struct('a',postsigma.c,'b',postsigma.C));
                res.sigma(:,k) = modreg.par.sigma;
            end
            
        else
            % select initial values for slice sampler
            %modify by plankensteiner begin
            init = [prior.init.beta(:,k)',prior.init.sigma(k)];
            % modify by plankensteiner end
            
            if sum(~data.censor(S==k))==0;
                log_likelihood = @(theta) sum(log(1-normcdf(yk, Xk*(theta(1:d)'), sqrt(theta(end)))));
                [log_likelihood, init] = Scaling(log_likelihood, init,'log_likelihood');
                if isempty(log_likelihood) res=[]; return; end
            else
                yc = yk(censor);
                Xc = Xk(censor,:);
                y = yk(~censor);
                X = Xk(~censor,:);
                log_likelihood = @(theta) sum(log(normpdf(y, (X*theta(1:d)'), sqrt(theta(end)))))+...
                    sum(log(1-normcdf(yc, (Xc*theta(1:d)'), sqrt(theta(end)))));
                [log_likelihood, init] = Scaling(log_likelihood, init,'log_likelihood');
                if isempty(log_likelihood) res=[]; return; end
            end
            
            log_posterior = @(par) log_likelihood(par)+priorBeta(par(1:d))+priorSigma(par(end));
            [log_posterior, init] = Scaling(log_posterior, init,'log_posterior');
            if isempty(log_posterior) res=[]; return; end
            
            trace = slicesample(init,Sperm.burnin+100,'logpdf',log_posterior,'width',4);
            
            sample = Sperm.burnin+ceil(rand()*(100));
            res.beta(:,k) = trace(sample,1:d)';
            res.sigma(:,k) = trace(sample,end)';
            
            % use a subsample of 30 draws to calculate the posterior parameters
            index = randperm(100);
            %                         indexchoose = Sperm.burnin+index(1:30);
            post.par.beta.b(:,k) = mean(trace(Sperm.burnin+index(1:30),1:d))';
            post.par.beta.B(:,:,k) = cov(trace(Sperm.burnin+index(1:30),1:d));
            post.par.beta.Binv(:,:,k) = inv(post.par.beta.B(:,:,k));
            
            parC = gamfit(1./trace(Sperm.burnin+index(1:30),end));
            post.par.sigma.c(:,k) = parC(1);
            post.par.sigma.C(:,k) = 1./parC(2);
            
            Sperm.sample{k,Sperm.index_perm}=trace;
        end
    end
else
    warn('slice sampler not implemented yet')
    res.error=true; return;
end
end



function [func, init] = Scaling(f,init,name)
% scaling of the function f
func = f;
scaling = 100;
% increase standard deviation by 1

while func(init)==0
    func = @(par) 10^scaling*f(par);
    scaling = scaling +100;
    if scaling ==400
        init = [zeros(1,size(init,2)-1),10];
        func = @(par) 10^100*f(par);
        if func(init)==0
            warn(['no slice sampling performed, ', name,' (init)=0'])
            func = []; return;
        else
            break;
        end
    end
end
end
